/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.encryption;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import org.apache.iceberg.encryption.Ciphers;
import org.apache.iceberg.io.IOUtil;
import org.apache.iceberg.io.SeekableInputStream;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;

public class AesGcmInputStream
extends SeekableInputStream {
    private final SeekableInputStream sourceStream;
    private final byte[] fileAADPrefix;
    private final Ciphers.AesGcmDecryptor decryptor;
    private final byte[] cipherBlockBuffer;
    private final byte[] currentPlainBlock;
    private final long numBlocks;
    private final int lastCipherBlockSize;
    private final long plainStreamSize;
    private final byte[] singleByte;
    private long plainStreamPosition;
    private long currentPlainBlockIndex;
    private int currentPlainBlockSize;

    AesGcmInputStream(SeekableInputStream sourceStream, long sourceLength, byte[] aesKey, byte[] fileAADPrefix) {
        this.sourceStream = sourceStream;
        this.fileAADPrefix = fileAADPrefix;
        this.decryptor = new Ciphers.AesGcmDecryptor(aesKey);
        this.cipherBlockBuffer = new byte[0x10001C];
        this.currentPlainBlock = new byte[0x100000];
        this.plainStreamPosition = 0L;
        this.currentPlainBlockIndex = -1L;
        this.currentPlainBlockSize = 0;
        long streamLength = sourceLength - (long)Ciphers.GCM_STREAM_HEADER_LENGTH;
        long numFullBlocks = Math.toIntExact(streamLength / 0x10001CL);
        long cipherFullBlockLength = numFullBlocks * 0x10001CL;
        int cipherBytesInLastBlock = Math.toIntExact(streamLength - cipherFullBlockLength);
        boolean fullBlocksOnly = 0 == cipherBytesInLastBlock;
        this.numBlocks = fullBlocksOnly ? numFullBlocks : numFullBlocks + 1L;
        this.lastCipherBlockSize = fullBlocksOnly ? 0x10001C : cipherBytesInLastBlock;
        long lastPlainBlockSize = (long)this.lastCipherBlockSize - 12L - 16L;
        this.plainStreamSize = numFullBlocks * 0x100000L + (fullBlocksOnly ? 0L : lastPlainBlockSize);
        this.singleByte = new byte[1];
    }

    private void validateHeader() throws IOException {
        byte[] headerBytes = new byte[Ciphers.GCM_STREAM_HEADER_LENGTH];
        IOUtil.readFully((InputStream)this.sourceStream, headerBytes, 0, headerBytes.length);
        Preconditions.checkState((boolean)Ciphers.GCM_STREAM_MAGIC.equals(ByteBuffer.wrap(headerBytes, 0, 4)), (Object)"Invalid GCM stream: magic does not match AGS1");
        int plainBlockSize = ByteBuffer.wrap(headerBytes, 4, 4).order(ByteOrder.LITTLE_ENDIAN).getInt();
        Preconditions.checkState((plainBlockSize == 0x100000 ? 1 : 0) != 0, (String)"Invalid GCM stream: block size %d != %d", (int)plainBlockSize, (int)0x100000);
    }

    public int available() {
        long maxAvailable = this.plainStreamSize - this.plainStreamPosition;
        if (maxAvailable >= Integer.MAX_VALUE) {
            return Integer.MAX_VALUE;
        }
        return (int)maxAvailable;
    }

    private int availableInCurrentBlock() {
        if (AesGcmInputStream.blockIndex(this.plainStreamPosition) != this.currentPlainBlockIndex) {
            return 0;
        }
        return this.currentPlainBlockSize - AesGcmInputStream.offsetInBlock(this.plainStreamPosition);
    }

    public int read(byte[] b, int off, int len) throws IOException {
        Preconditions.checkArgument((len >= 0 ? 1 : 0) != 0, (Object)("Invalid read length: " + len));
        if (this.currentPlainBlockIndex < 0L) {
            this.decryptBlock(0L);
        }
        if (this.available() <= 0 && len > 0) {
            return -1;
        }
        if (len == 0) {
            return 0;
        }
        int totalBytesRead = 0;
        int resultBufferOffset = off;
        int remainingBytesToRead = len;
        while (remainingBytesToRead > 0) {
            int availableInBlock = this.availableInCurrentBlock();
            if (availableInBlock > 0) {
                int bytesToCopy = Math.min(availableInBlock, remainingBytesToRead);
                int offsetInBlock = AesGcmInputStream.offsetInBlock(this.plainStreamPosition);
                System.arraycopy(this.currentPlainBlock, offsetInBlock, b, resultBufferOffset, bytesToCopy);
                totalBytesRead += bytesToCopy;
                remainingBytesToRead -= bytesToCopy;
                resultBufferOffset += bytesToCopy;
                this.plainStreamPosition += (long)bytesToCopy;
                continue;
            }
            if (this.available() <= 0) break;
            this.decryptBlock(AesGcmInputStream.blockIndex(this.plainStreamPosition));
        }
        return totalBytesRead > 0 ? totalBytesRead : -1;
    }

    public void seek(long newPos) throws IOException {
        if (newPos < 0L) {
            throw new IOException("Invalid position: " + newPos);
        }
        if (newPos > this.plainStreamSize) {
            throw new EOFException("Invalid position: " + newPos + " > stream length, " + this.plainStreamSize);
        }
        this.plainStreamPosition = newPos;
    }

    public long skip(long n) {
        if (n <= 0L) {
            return 0L;
        }
        long bytesLeftInStream = this.plainStreamSize - this.plainStreamPosition;
        if (n > bytesLeftInStream) {
            this.plainStreamPosition = this.plainStreamSize;
            return bytesLeftInStream;
        }
        this.plainStreamPosition += n;
        return n;
    }

    public long getPos() throws IOException {
        return this.plainStreamPosition;
    }

    public int read() throws IOException {
        int read = this.read(this.singleByte);
        if (read == -1) {
            return -1;
        }
        return this.singleByte[0] >= 0 ? this.singleByte[0] : 256 + this.singleByte[0];
    }

    public void close() throws IOException {
        this.sourceStream.close();
    }

    private void decryptBlock(long blockIndex) throws IOException {
        if (blockIndex == this.currentPlainBlockIndex) {
            return;
        }
        long blockPositionInStream = AesGcmInputStream.blockOffset(blockIndex);
        if (this.sourceStream.getPos() != blockPositionInStream) {
            if (this.sourceStream.getPos() == 0L) {
                this.validateHeader();
            }
            this.sourceStream.seek(blockPositionInStream);
        }
        boolean isLastBlock = blockIndex == this.numBlocks - 1L;
        int cipherBlockSize = isLastBlock ? this.lastCipherBlockSize : 0x10001C;
        IOUtil.readFully((InputStream)this.sourceStream, this.cipherBlockBuffer, 0, cipherBlockSize);
        byte[] blockAAD = Ciphers.streamBlockAAD(this.fileAADPrefix, Math.toIntExact(blockIndex));
        this.decryptor.decrypt(this.cipherBlockBuffer, 0, cipherBlockSize, this.currentPlainBlock, 0, blockAAD);
        this.currentPlainBlockSize = cipherBlockSize - 12 - 16;
        this.currentPlainBlockIndex = blockIndex;
    }

    private static long blockIndex(long plainPosition) {
        return plainPosition / 0x100000L;
    }

    private static int offsetInBlock(long plainPosition) {
        return Math.toIntExact(plainPosition % 0x100000L);
    }

    private static long blockOffset(long blockIndex) {
        return blockIndex * 0x10001CL + (long)Ciphers.GCM_STREAM_HEADER_LENGTH;
    }

    static long calculatePlaintextLength(long sourceLength) {
        long streamLength = sourceLength - (long)Ciphers.GCM_STREAM_HEADER_LENGTH;
        if (streamLength == 0L) {
            return 0L;
        }
        long numberOfFullBlocks = streamLength / 0x10001CL;
        long fullBlockSize = numberOfFullBlocks * 0x10001CL;
        long cipherBytesInLastBlock = streamLength - fullBlockSize;
        boolean fullBlocksOnly = 0L == cipherBytesInLastBlock;
        long plainBytesInLastBlock = fullBlocksOnly ? 0L : cipherBytesInLastBlock - 12L - 16L;
        return numberOfFullBlocks * 0x100000L + plainBytesInLastBlock;
    }
}

