/*****************************************************************************
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * 
 ****************************************************************************/

package org.apache.xmpbox.schema;

import java.io.ByteArrayOutputStream;
import java.util.List;
import javax.xml.transform.TransformerException;

import org.apache.xmpbox.XMPMetadata;
import org.apache.xmpbox.type.BadFieldValueException;
import org.apache.xmpbox.type.JobType;
import org.apache.xmpbox.type.StructuredType;
import org.apache.xmpbox.xml.DomXmpParser;
import org.apache.xmpbox.xml.XmpParsingException;
import org.apache.xmpbox.xml.XmpSerializer;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import org.junit.Test;

public class BasicJobTicketSchemaTest
{
    @Test
    public void testAddTwoJobs() throws XmpParsingException, TransformerException, BadFieldValueException
    {
        XMPMetadata metadata = XMPMetadata.createXMPMetadata();
        XmpSerializer serializer = new XmpSerializer();
        DomXmpParser builder = new DomXmpParser();

        XMPBasicJobTicketSchema basic = metadata.createAndAddBasicJobTicketSchema();
        basic.addJob("zeid1", "zename1", "zeurl1", "aaa");
        basic.addJob("zeid2", "zename2", "zeurl2");

        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        serializer.serialize(metadata, bos, true);

        XMPMetadata rxmp = builder.parse(bos.toByteArray());

        XMPBasicJobTicketSchema jt = rxmp.getBasicJobTicketSchema();
        List<JobType> jobs = jt.getJobs();
        assertEquals(2, jobs.size());
        JobType jt0 = jobs.get(0);
        assertEquals("zeid1", jt0.getId());
        assertEquals("zename1", jt0.getName());
        assertEquals("zeurl1", jt0.getUrl());
        JobType jt1 = jobs.get(1);
        assertEquals("zeid2", jt1.getId());
        assertEquals("zename2", jt1.getName());
        assertEquals("zeurl2", jt1.getUrl());
    }

    @Test
    public void testAddWithDefaultPrefix() throws XmpParsingException, TransformerException, BadFieldValueException
    {
        XMPMetadata metadata = XMPMetadata.createXMPMetadata();
        XmpSerializer serializer = new XmpSerializer();
        DomXmpParser builder = new DomXmpParser();

        XMPBasicJobTicketSchema basic = metadata.createAndAddBasicJobTicketSchema();

        basic.addJob("zeid2", "zename2", "zeurl2");

        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        serializer.serialize(metadata, bos, true);
        XMPMetadata rxmp = builder.parse(bos.toByteArray());

        XMPBasicJobTicketSchema jt = rxmp.getBasicJobTicketSchema();
        assertNotNull(jt);
        assertEquals(1, jt.getJobs().size());

        JobType job = jt.getJobs().get(0);
        assertEquals("zeid2", job.getId());
        assertEquals("zename2", job.getName());
        assertEquals("zeurl2", job.getUrl());
    }

    @Test
    public void testAddWithDefinedPrefix() throws TransformerException, XmpParsingException, BadFieldValueException
    {
        XMPMetadata metadata = XMPMetadata.createXMPMetadata();
        XmpSerializer serializer = new XmpSerializer();
        DomXmpParser builder = new DomXmpParser();

        XMPBasicJobTicketSchema basic = metadata.createAndAddBasicJobTicketSchema();

        basic.addJob("zeid2", "zename2", "zeurl2", "aaa");

        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        serializer.serialize(metadata, bos, true);
        XMPMetadata rxmp = builder.parse(bos.toByteArray());

        XMPBasicJobTicketSchema jt = rxmp.getBasicJobTicketSchema();
        assertEquals(1, jt.getJobs().size());

        JobType job = jt.getJobs().get(0);

        StructuredType stjob = JobType.class.getAnnotation(StructuredType.class);

        assertEquals("zeid2", job.getId());
        assertEquals("zename2", job.getName());
        assertEquals("zeurl2", job.getUrl());

        assertEquals(stjob.namespace(), job.getNamespace());
        assertEquals("aaa", job.getPrefix());
    }

}
